<?php
// userAdd.php — Panel independiente para alta y administración de usuarios internos / clientes / artículos
// Panel independiente de la PWA (no usa current_user() para autenticarse)
// Acceso basado en la tabla `users` (roles admin / superadmin)

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/auth.php';

// Cargar helper de teléfono si existe (mismo que usa index.php)
$phoneHelper = __DIR__ . '/../includes/phone_helper.php';
if (is_file($phoneHelper)) require_once $phoneHelper;

if (session_status() !== PHP_SESSION_ACTIVE) {
    session_start();
}

header('Content-Type: text/html; charset=utf-8');

$pdo = pdo();

/* =========================
   Helpers de teléfono (robustos)
========================= */
function digits_only(string $s): string {
    return preg_replace('/\D+/', '', $s) ?? '';
}

function looks_like_username(string $raw): bool {
    // Si trae letras o símbolos típicos de usuario, NO lo tratamos como teléfono
    return (bool)preg_match('/[A-Za-z@_]/', $raw);
}

/**
 * Normaliza a E.164 con regla requerida:
 * MX => +521 + 10 dígitos
 * US => +1   + 10 dígitos
 * Si ya viene con +, lo conserva (limpia a solo dígitos y re-agrega +).
 */
function normalize_phone_e164_custom(string $country, string $raw): string {
    $raw = trim($raw);
    if ($raw === '') return '';

    // Si es usuario (no teléfono), devolver tal cual
    if (looks_like_username($raw)) return $raw;

    // Si ya viene con +, limpiar
    if (strpos($raw, '+') === 0) {
        $d = digits_only($raw);
        return ($d !== '') ? ('+' . $d) : '';
    }

    $d = digits_only($raw);
    if ($d === '') return '';

    $country = strtoupper(trim($country ?: 'MX'));

    if ($country === 'MX') {
        // Caso ideal: 10 dígitos nacionales
        if (strlen($d) === 10) return '+521' . $d;

        // Si viene como 52 + 10 dígitos
        if (strlen($d) === 12 && substr($d, 0, 2) === '52') {
            $n = substr($d, 2); // 10 dígitos
            return '+521' . $n;
        }

        // Si viene como 521 + 10 dígitos
        if (strlen($d) === 13 && substr($d, 0, 3) === '521') {
            return '+' . $d;
        }

        // Fallback conservador: si trae 11–15 dígitos, lo tratamos como internacional
        if (strlen($d) >= 11 && strlen($d) <= 15) return '+' . $d;

        return '';
    }

    if ($country === 'US') {
        // 10 dígitos nacionales
        if (strlen($d) === 10) return '+1' . $d;

        // 11 dígitos empezando con 1
        if (strlen($d) === 11 && substr($d, 0, 1) === '1') return '+' . $d;

        if (strlen($d) >= 11 && strlen($d) <= 15) return '+' . $d;

        return '';
    }

    // Genérico
    if (strlen($d) >= 11 && strlen($d) <= 15) return '+' . $d;
    return '';
}

/** Devuelve últimos 10 dígitos (para phone_national si lo usas). */
function phone_national_10_custom(string $raw): string {
    $d = digits_only($raw);
    if ($d === '') return '';
    return (strlen($d) >= 10) ? substr($d, -10) : $d;
}

function table_has_column_local(PDO $pdo, string $table, string $column): bool {
    static $cache = [];
    $key = strtolower($table . '.' . $column);
    if (isset($cache[$key])) return (bool)$cache[$key];

    $stmt = $pdo->prepare("
        SELECT COUNT(*)
        FROM information_schema.columns
        WHERE table_schema = DATABASE()
          AND table_name = ?
          AND column_name = ?
    ");
    $stmt->execute([$table, $column]);
    $ok = ((int)$stmt->fetchColumn() > 0);
    $cache[$key] = $ok;
    return $ok;
}

/**
 * Para login del panel: genera candidatos si el usuario escribió solo 10 dígitos, etc.
 * Esto permite que el admin escriba "4441234567" y encuentre un registro guardado como "+5214441234567".
 */
function phone_login_candidates(string $input): array {
    $input = trim($input);
    if ($input === '') return [];

    // Si parece username, no inventamos formatos
    if (looks_like_username($input)) return [$input];

    // Si trae +, respetar
    if (strpos($input, '+') === 0) {
        $d = digits_only($input);
        return $d ? ['+' . $d] : [];
    }

    $d = digits_only($input);
    if ($d === '') return [$input];

    $cand = [];

    // Exacto como lo escribió (por si guardaron sin +)
    $cand[] = $input;
    $cand[] = $d;

    // 10 dígitos: probar MX y US
    if (strlen($d) === 10) {
        $cand[] = '+521' . $d;
        $cand[] = '+52'  . $d; // por compatibilidad histórica
        $cand[] = '+1'   . $d;
    }

    // 12 dígitos comenzando 52 => MX
    if (strlen($d) === 12 && substr($d, 0, 2) === '52') {
        $n = substr($d, 2);
        $cand[] = '+52' . $n;
        $cand[] = '+521' . $n;
    }

    // 13 dígitos comenzando 521
    if (strlen($d) === 13 && substr($d, 0, 3) === '521') {
        $cand[] = '+' . $d;
    }

    // 11 dígitos comenzando 1
    if (strlen($d) === 11 && substr($d, 0, 1) === '1') {
        $cand[] = '+' . $d;
    }

    // Uniq
    $cand = array_values(array_unique(array_filter($cand, fn($x) => is_string($x) && $x !== '')));
    return $cand;
}

/* =====================================================
   1) LOGIN PROPIO DEL PANEL BASADO EN TABLA `users`
===================================================== */

$panelUserId   = $_SESSION['panel_user_id']   ?? null;
$panelUserName = $_SESSION['panel_user_name'] ?? '';
$panelUserRole = $_SESSION['panel_user_role'] ?? '';
$panelOk       = $panelUserId ? true : false;

$loginErr = '';

if (isset($_GET['logout_panel'])) {
    unset($_SESSION['panel_user_id'], $_SESSION['panel_user_name'], $_SESSION['panel_user_role']);
    header('Location: userAdd.php');
    exit;
}

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['panel_login'])) {
    $userPhone = trim($_POST['panel_user'] ?? '');
    $pass      = $_POST['panel_pass'] ?? '';

    if ($userPhone === '' || $pass === '') {
        $loginErr = 'Captura teléfono/usuario y contraseña.';
    } else {
        $cand = phone_login_candidates($userPhone);
        if (!$cand) $cand = [$userPhone];

        $in = implode(',', array_fill(0, count($cand), '?'));

        $stmt = $pdo->prepare("
            SELECT id, name, phone, role, password_hash
            FROM users
            WHERE phone IN ($in)
            LIMIT 1
        ");
        $stmt->execute($cand);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$row) {
            $loginErr = 'Usuario no encontrado en la base de datos.';
        } elseif (!password_verify($pass, (string)$row['password_hash'])) {
            $loginErr = 'Contraseña incorrecta.';
        } else {
            $rolesPanel = ['admin', 'superadmin'];
            if (!in_array((string)$row['role'], $rolesPanel, true)) {
                $loginErr = 'Este usuario no tiene permisos para el panel.';
            } else {
                $_SESSION['panel_user_id']   = (int)$row['id'];
                $_SESSION['panel_user_name'] = (string)$row['name'];
                $_SESSION['panel_user_role'] = (string)$row['role'];
                $panelUserId   = (int)$row['id'];
                $panelUserName = (string)$row['name'];
                $panelUserRole = (string)$row['role'];
                $panelOk       = true;
            }
        }
    }
}

if (!$panelOk) {
?>
<!doctype html>
<html lang="es">
<head>
  <meta charset="utf-8">
  <title>Acceso panel de usuarios · Ventas Blanquita</title>
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <link rel="stylesheet" href="assets/style.css">
  <style>
    html, body { height: 100%; }
    body {
      background: #f3f4f6;
      display: flex;
      align-items: center;
      justify-content: center;
    }
    .login-wrapper { width: 100%; max-width: 420px; padding: 1rem; }
  </style>
</head>
<body>
  <div class="login-wrapper">
    <div class="card shadow-sm border-0">
      <div class="card-body">
        <h1 class="h5 text-primary mb-3">Panel de administración · Ventas Blanquita</h1>
        <p class="text-muted small">
          Acceso restringido. Este panel es independiente de la PWA y solo debe ser usado por personal autorizado.
        </p>

        <?php if ($loginErr): ?>
          <div class="alert alert-danger py-2 small mb-3"><?= h($loginErr) ?></div>
        <?php endif; ?>

        <form method="post" autocomplete="off">
          <input type="hidden" name="panel_login" value="1">

          <div class="mb-3">
            <label class="form-label">Teléfono / Usuario</label>
            <input type="text" name="panel_user" class="form-control" required>
            <div class="form-text">
              Puedes escribir 10 dígitos y el sistema buscará variantes (+521 / +52 / +1) automáticamente.
            </div>
          </div>

          <div class="mb-3">
            <label class="form-label">Contraseña</label>
            <input type="password" name="panel_pass" class="form-control" required>
          </div>

          <button class="btn btn-primary w-100">Entrar al panel</button>
        </form>

        <p class="small text-muted mt-3 mb-0">
          El usuario debe existir en la tabla <code>users</code> y tener rol <strong>admin</strong> o <strong>superadmin</strong>.
        </p>
      </div>
    </div>
  </div>

  <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
<?php
    exit;
}

/* =====================================================
   2) PANEL (YA LOGUEADO AL PANEL)
===================================================== */

$msg = '';
$err = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    $action = (string)$_POST['action'];

    if ($action === 'crear_usuario') {
        $name     = trim($_POST['name'] ?? '');
        $country  = strtoupper(trim($_POST['country'] ?? 'MX'));
        $phoneRaw = trim($_POST['phone'] ?? '');
        $role     = trim($_POST['role'] ?? '');
        $password = (string)($_POST['password'] ?? '');
        $address  = trim($_POST['address'] ?? '');
        $cp       = trim($_POST['postal_code'] ?? '');

        $rolesPermitidos = ['ventas', 'repartidor', 'admin', 'tienda'];

        if (!$name || !$phoneRaw || !$password || !$role) {
            $err = 'Completa todos los campos obligatorios del usuario interno.';
        } elseif (!in_array($role, $rolesPermitidos, true)) {
            $err = 'Rol no permitido para usuario interno.';
        } else {
            // Normalizar teléfono según país (si NO es username)
            $phoneFinal = normalize_phone_e164_custom($country, $phoneRaw);
            if ($phoneFinal === '') {
                $err = 'Teléfono inválido. Para MX/US captura 10 dígitos (sin espacios).';
            } else {
                try {
                    if ($cp === '') $cp = '00000';

                    $nat10 = looks_like_username($phoneRaw) ? '' : phone_national_10_custom($phoneRaw);

                    // Llamar register_user con firma compatible
                    $createdOk = false;
                    if (function_exists('register_user')) {
                        try {
                            $rf = new ReflectionFunction('register_user');
                            $n  = $rf->getNumberOfParameters();

                            if ($n >= 8) {
                                // register_user($name, $phone, $address, $password, $cp, $role, $phone_country, $nat10)
                                register_user($name, $phoneFinal, $address, $password, $cp, $role, $country, $nat10);
                                $createdOk = true;
                            } else {
                                // Firma antigua (mínima)
                                register_user($name, $phoneFinal, $address, $password, $cp);
                                $createdOk = true;
                            }
                        } catch (Throwable $e) {
                            // Fallback extra seguro: intentar firma mínima
                            register_user($name, $phoneFinal, $address, $password, $cp);
                            $createdOk = true;
                        }
                    }

                    if (!$createdOk) {
                        throw new Exception('register_user() no disponible');
                    }

                    // Forzar rol solicitado y activar usuario
                    $updCols = "role = ?, is_active = 1";
                    $updVals = [$role];

                    // Guardar country/national si existen columnas
                    if (!looks_like_username($phoneRaw)) {
                        if (table_has_column_local($pdo, 'users', 'phone_country')) {
                            $updCols .= ", phone_country = ?";
                            $updVals[] = $country;
                        }
                        if (table_has_column_local($pdo, 'users', 'phone_national')) {
                            $updCols .= ", phone_national = ?";
                            $updVals[] = $nat10;
                        }
                    }

                    $updVals[] = $phoneFinal;

                    $stmt = $pdo->prepare("UPDATE users SET {$updCols} WHERE phone = ?");
                    $stmt->execute($updVals);

                    $msg = 'Usuario interno creado correctamente con rol: ' . h($role)
                         . (looks_like_username($phoneRaw) ? '' : (' · Tel: ' . h($phoneFinal)));
                } catch (Exception $e) {
                    $err = 'No se pudo crear el usuario (¿teléfono duplicado?).';
                }
            }
        }
    }

    if ($action === 'toggle_user' && isset($_POST['user_id'])) {
        $uid = (int)$_POST['user_id'];

        if ($uid === (int)$panelUserId) {
            $err = 'No puedes desactivar tu propio usuario del panel.';
        } else {
            $stmt = $pdo->prepare("SELECT is_active FROM users WHERE id = ? AND role IN ('ventas','repartidor','admin','tienda')");
            $stmt->execute([$uid]);
            $row = $stmt->fetch(PDO::FETCH_ASSOC);

            if ($row) {
                $new = (int)$row['is_active'] ? 0 : 1;
                $upd = $pdo->prepare("UPDATE users SET is_active = ? WHERE id = ?");
                $upd->execute([$new, $uid]);
                $msg = 'Estado del usuario interno actualizado.';
            }
        }
    }

    if ($action === 'toggle_cliente' && isset($_POST['user_id'])) {
        $uid = (int)$_POST['user_id'];

        $stmt = $pdo->prepare("SELECT is_active FROM users WHERE id = ? AND role = 'cliente'");
        $stmt->execute([$uid]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);

        if ($row) {
            $new = (int)$row['is_active'] ? 0 : 1;
            $upd = $pdo->prepare("UPDATE users SET is_active = ? WHERE id = ?");
            $upd->execute([$new, $uid]);
            $msg = 'Estado del cliente actualizado.';
        }
    }

    if ($action === 'crear_articulo') {
        $nombre = trim($_POST['product_name'] ?? '');
        $precio = trim($_POST['product_price'] ?? '');

        if ($nombre === '' || $precio === '') {
            $err = 'Captura nombre y precio del artículo.';
        } elseif (!is_numeric($precio) || $precio < 0) {
            $err = 'Precio inválido.';
        } else {
            $stmt = $pdo->prepare("
                INSERT INTO products (name, price, is_active, created_at)
                VALUES (?, ?, 1, NOW())
            ");
            $stmt->execute([$nombre, $precio]);
            $msg = 'Artículo registrado correctamente.';
        }
    }

    if ($action === 'update_articulo' && isset($_POST['product_id'])) {
        $pid    = (int)($_POST['product_id'] ?? 0);
        $nombre = trim($_POST['product_name'] ?? '');
        $precio = trim($_POST['product_price'] ?? '');

        if ($pid <= 0) {
            $err = 'Artículo no válido.';
        } elseif ($nombre === '' || $precio === '') {
            $err = 'Captura nombre y precio del artículo a actualizar.';
        } elseif (!is_numeric($precio) || $precio < 0) {
            $err = 'Precio inválido en la actualización.';
        } else {
            $stmt = $pdo->prepare("UPDATE products SET name = ?, price = ? WHERE id = ?");
            $stmt->execute([$nombre, $precio, $pid]);
            $msg = 'Artículo actualizado correctamente.';
        }
    }

    if ($action === 'toggle_articulo' && isset($_POST['product_id'])) {
        $pid = (int)$_POST['product_id'];

        $stmt = $pdo->prepare("SELECT is_active FROM products WHERE id = ?");
        $stmt->execute([$pid]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);

        if ($row) {
            $new = (int)$row['is_active'] ? 0 : 1;
            $upd = $pdo->prepare("UPDATE products SET is_active = ? WHERE id = ?");
            $upd->execute([$new, $pid]);
            $msg = 'Estado del artículo actualizado.';
        }
    }
}

/* =====================================================
   3) CONSULTAS PARA MOSTRAR DATOS EN EL PANEL
===================================================== */

$usuariosInternos = $pdo->query("
    SELECT id, name, phone, role,
           IFNULL(is_active, 1) AS is_active,
           created_at
    FROM users
    WHERE role IN ('ventas','repartidor','admin','tienda')
    ORDER BY created_at DESC
")->fetchAll(PDO::FETCH_ASSOC);

$clientes = $pdo->query("
    SELECT id, name, phone,
           IFNULL(is_active, 1) AS is_active,
           created_at
    FROM users
    WHERE role = 'cliente'
    ORDER BY created_at DESC
    LIMIT 200
")->fetchAll(PDO::FETCH_ASSOC);

$productos = $pdo->query("
    SELECT id, name, price,
           IFNULL(is_active, 1) AS is_active,
           created_at
    FROM products
    ORDER BY name ASC
")->fetchAll(PDO::FETCH_ASSOC);

$editProduct   = null;
$editProductId = null;
if (isset($_GET['edit_product'])) {
    $editProductId = (int)$_GET['edit_product'];
} elseif ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['action'] ?? '') === 'update_articulo' && isset($_POST['product_id'])) {
    $editProductId = (int)$_POST['product_id'];
}

if ($editProductId) {
    $stmt = $pdo->prepare("SELECT id, name, price FROM products WHERE id = ?");
    $stmt->execute([$editProductId]);
    $editProduct = $stmt->fetch(PDO::FETCH_ASSOC) ?: null;
}
?>
<!doctype html>
<html lang="es">
<head>
  <meta charset="utf-8">
  <title>Panel de usuarios internos · Ventas Blanquita</title>
  <meta name="viewport" content="width=device-width, initial-scale=1">

  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <link rel="stylesheet" href="assets/style.css">

  <style>
    html, body { height: 100%; }
    body { background: #f3f4f6; }
    .card-panel {
      border-radius: 18px;
      box-shadow: 0 12px 30px rgba(15,23,42,.12);
      border: 0;
    }
    .nav-tabs-wrapper {
      overflow-x: auto;
      -webkit-overflow-scrolling: touch;
      scrollbar-width: thin;
    }
    .nav-tabs {
      white-space: nowrap;
      flex-wrap: nowrap;
      border-bottom: 1px solid #e5e7eb;
    }
    .nav-tabs .nav-link {
      border: none;
      border-bottom: 2px solid transparent;
      color: #64748b;
    }
    .nav-tabs .nav-link.active {
      color: #1d4ed8;
      border-bottom-color: #1d4ed8;
      background-color: transparent;
    }
    .table-sm th, .table-sm td {
      padding-top: .35rem;
      padding-bottom: .35rem;
    }
    .badge-estado { font-size: .75rem; }

    @media (max-width: 767.98px) {
      .navbar .navbar-brand { font-size: .9rem; }
      .navbar .btn { font-size: .75rem; padding: .25rem .5rem; }
      .card-panel { border-radius: 0; box-shadow: none; }
      .container-fluid { padding-left: .5rem; padding-right: .5rem; }
      .table-responsive { max-height: none !important; }
    }
  </style>
</head>
<body>

<nav class="navbar navbar-expand-lg navbar-dark bg-primary">
  <div class="container-fluid">
    <span class="navbar-brand">Ventas Blanquita · Panel de administración</span>
    <button class="navbar-toggler" type="button" data-bs-toggle="collapse" data-bs-target="#navPanel"
            aria-controls="navPanel" aria-expanded="false" aria-label="Toggle navigation">
      <span class="navbar-toggler-icon"></span>
    </button>
    <div class="collapse navbar-collapse justify-content-end" id="navPanel">
      <div class="d-flex flex-column flex-sm-row align-items-sm-center mt-2 mt-sm-0">
        <span class="text-white-50 small me-sm-3 mb-2 mb-sm-0">
          <?= 'Conectado como: ' . h($panelUserName) . ' (' . h($panelUserRole) . ')' ?>
        </span>
        <div class="btn-group btn-group-sm">
          <a href="index.php" class="btn btn-outline-light">PWA pedidos</a>
          <a href="?logout_panel=1" class="btn btn-outline-light">Salir</a>
        </div>
      </div>
    </div>
  </div>
</nav>

<div class="container-fluid py-3 py-md-4">
  <div class="row justify-content-center">
    <div class="col-12 col-xl-11">
      <p class="text-muted small mb-3">
        Este panel es independiente de la PWA. Solo personal autorizado debe tener acceso.
      </p>

      <?php if ($msg): ?>
        <div class="alert alert-success py-2"><?= h($msg) ?></div>
      <?php endif; ?>
      <?php if ($err): ?>
        <div class="alert alert-danger py-2"><?= h($err) ?></div>
      <?php endif; ?>

      <div class="card card-panel">
        <div class="card-body">
          <div class="nav-tabs-wrapper mb-3">
            <ul class="nav nav-tabs" id="panelTabs" role="tablist">
              <li class="nav-item" role="presentation">
                <button class="nav-link active" id="tab-usuarios-tab" data-bs-toggle="tab"
                        data-bs-target="#tab-usuarios" type="button" role="tab">
                  Usuarios internos
                </button>
              </li>
              <li class="nav-item" role="presentation">
                <button class="nav-link" id="tab-clientes-tab" data-bs-toggle="tab"
                        data-bs-target="#tab-clientes" type="button" role="tab">
                  Clientes
                </button>
              </li>
              <li class="nav-item" role="presentation">
                <button class="nav-link" id="tab-articulos-tab" data-bs-toggle="tab"
                        data-bs-target="#tab-articulos" type="button" role="tab">
                  Artículos
                </button>
              </li>
              <li class="nav-item" role="presentation">
                <button class="nav-link" id="tab-reset-tab" data-bs-toggle="tab"
                        data-bs-target="#tab-reset" type="button" role="tab">
                  Reset de contraseña
                </button>
              </li>
            </ul>
          </div>

          <div class="tab-content" id="panelTabsContent">
            <!-- ================= TAB: USUARIOS INTERNOS ================= -->
            <div class="tab-pane fade show active" id="tab-usuarios" role="tabpanel">
              <div class="row g-3">
                <div class="col-12 col-lg-4">
                  <form method="post" class="card card-body border-0 shadow-sm">
                    <h2 class="h6 mb-3">Nuevo usuario interno</h2>
                    <input type="hidden" name="action" value="crear_usuario">

                    <div class="mb-2">
                      <label class="form-label">Nombre</label>
                      <input type="text" name="name" class="form-control" required>
                    </div>

                    <div class="mb-2">
                      <label class="form-label">País (para teléfonos)</label>
                      <select name="country" class="form-select" required>
                        <option value="MX" selected>México (+52)</option>
                        <option value="US">USA (+1)</option>
                      </select>
                      <div class="form-text">Para MX captura 10 dígitos y se guardará como <strong>+521</strong> + 10 dígitos.</div>
                    </div>

                    <div class="mb-2">
                      <label class="form-label">Teléfono / Usuario</label>
                      <input type="text" name="phone" class="form-control" required placeholder="10 dígitos o usuario">
                      <div class="form-text">Si escribes solo dígitos (10), se normaliza automáticamente.</div>
                    </div>

                    <div class="mb-2">
                      <label class="form-label">Rol</label>
                      <select name="role" class="form-select" required>
                        <option value="">-- Selecciona rol --</option>
                        <option value="ventas">Ventas</option>
                        <option value="repartidor">Repartidor</option>
                        <option value="admin">Administrador</option>
                        <option value="tienda">Tienda</option>
                      </select>
                    </div>

                    <div class="mb-2">
                      <label class="form-label">Contraseña</label>
                      <input type="password" name="password" class="form-control" required>
                    </div>

                    <div class="mb-2">
                      <label class="form-label">Dirección (opcional)</label>
                      <textarea name="address" class="form-control" rows="2"></textarea>
                    </div>

                    <div class="mb-3">
                      <label class="form-label">Código postal (opcional)</label>
                      <input type="text" name="postal_code" class="form-control" maxlength="5"
                             pattern="\d{5}" placeholder="Ej. 78399">
                    </div>

                    <button class="btn btn-primary w-100">Guardar usuario</button>
                  </form>
                </div>

                <div class="col-12 col-lg-8">
                  <div class="card card-body border-0 shadow-sm h-100">
                    <h2 class="h6 mb-3">Usuarios internos (ventas / repartidor / admin / tienda)</h2>
                    <?php if (!$usuariosInternos): ?>
                      <p class="small text-muted mb-0">Aún no hay usuarios internos registrados.</p>
                    <?php else: ?>
                      <div class="table-responsive" style="max-height: 380px; overflow:auto;">
                        <table class="table table-sm align-middle mb-0">
                          <thead>
                            <tr>
                              <th>ID</th>
                              <th>Nombre</th>
                              <th>Teléfono</th>
                              <th>Rol</th>
                              <th>Estado</th>
                              <th>Alta</th>
                              <th></th>
                            </tr>
                          </thead>
                          <tbody>
                            <?php foreach ($usuariosInternos as $usr): ?>
                              <tr>
                                <td><?= h($usr['id']) ?></td>
                                <td><?= h($usr['name']) ?></td>
                                <td><?= h($usr['phone']) ?></td>
                                <td><?= h($usr['role']) ?></td>
                                <td>
                                  <?php if ((int)$usr['is_active']): ?>
                                    <span class="badge bg-success badge-estado">Activo</span>
                                  <?php else: ?>
                                    <span class="badge bg-secondary badge-estado">Inactivo</span>
                                  <?php endif; ?>
                                </td>
                                <td class="small text-muted"><?= h($usr['created_at']) ?></td>
                                <td>
                                  <form method="post" class="d-inline">
                                    <input type="hidden" name="action" value="toggle_user">
                                    <input type="hidden" name="user_id" value="<?= (int)$usr['id'] ?>">
                                    <?php if ((int)$usr['is_active']): ?>
                                      <button class="btn btn-sm btn-outline-danger">Desactivar</button>
                                    <?php else: ?>
                                      <button class="btn btn-sm btn-outline-success">Activar</button>
                                    <?php endif; ?>
                                  </form>
                                </td>
                              </tr>
                            <?php endforeach; ?>
                          </tbody>
                        </table>
                      </div>
                    <?php endif; ?>
                  </div>
                </div>
              </div>
            </div>

            <!-- ================= TAB: CLIENTES ================= -->
            <div class="tab-pane fade" id="tab-clientes" role="tabpanel">
              <div class="row">
                <div class="col-12">
                  <div class="card card-body border-0 shadow-sm">
                    <h2 class="h6 mb-2">Clientes registrados (rol cliente)</h2>
                    <p class="small text-muted">
                      Se muestran los últimos 200 clientes para evitar que la tabla sea muy grande.
                    </p>
                    <?php if (!$clientes): ?>
                      <p class="small text-muted mb-0">Aún no hay clientes registrados.</p>
                    <?php else: ?>
                      <div class="table-responsive" style="max-height: 430px; overflow:auto;">
                        <table class="table table-sm align-middle mb-0">
                          <thead>
                            <tr>
                              <th>ID</th>
                              <th>Nombre</th>
                              <th>Teléfono</th>
                              <th>Estado</th>
                              <th>Alta</th>
                              <th></th>
                            </tr>
                          </thead>
                          <tbody>
                            <?php foreach ($clientes as $cl): ?>
                              <tr>
                                <td><?= h($cl['id']) ?></td>
                                <td><?= h($cl['name']) ?></td>
                                <td><?= h($cl['phone']) ?></td>
                                <td>
                                  <?php if ((int)$cl['is_active']): ?>
                                    <span class="badge bg-success badge-estado">Activo</span>
                                  <?php else: ?>
                                    <span class="badge bg-secondary badge-estado">Inactivo</span>
                                  <?php endif; ?>
                                </td>
                                <td class="small text-muted"><?= h($cl['created_at']) ?></td>
                                <td>
                                  <form method="post" class="d-inline">
                                    <input type="hidden" name="action" value="toggle_cliente">
                                    <input type="hidden" name="user_id" value="<?= (int)$cl['id'] ?>">
                                    <?php if ((int)$cl['is_active']): ?>
                                      <button class="btn btn-sm btn-outline-danger">Desactivar</button>
                                    <?php else: ?>
                                      <button class="btn btn-sm btn-outline-success">Activar</button>
                                    <?php endif; ?>
                                  </form>
                                </td>
                              </tr>
                            <?php endforeach; ?>
                          </tbody>
                        </table>
                      </div>
                    <?php endif; ?>
                  </div>
                </div>
              </div>
            </div>

            <!-- ================= TAB: ARTÍCULOS ================= -->
            <div class="tab-pane fade" id="tab-articulos" role="tabpanel">
              <div class="row g-3">
                <div class="col-12 col-lg-4">
                  <form method="post" class="card card-body border-0 shadow-sm">
                    <h2 class="h6 mb-3">Alta de artículos</h2>
                    <input type="hidden" name="action" value="crear_articulo">

                    <div class="mb-2">
                      <label class="form-label">Nombre del artículo</label>
                      <input type="text" name="product_name" class="form-control" required>
                    </div>

                    <div class="mb-3">
                      <label class="form-label">Precio (MXN)</label>
                      <input type="number" step="0.01" min="0" name="product_price" class="form-control" required>
                    </div>

                    <button class="btn btn-primary w-100">Guardar artículo</button>
                  </form>

                  <?php if ($editProduct): ?>
                    <form method="post" class="card card-body border-0 shadow-sm mt-3">
                      <h2 class="h6 mb-3">
                        Editar artículo ID <?= h($editProduct['id']) ?>
                      </h2>
                      <input type="hidden" name="action" value="update_articulo">
                      <input type="hidden" name="product_id" value="<?= (int)$editProduct['id'] ?>">

                      <div class="mb-2">
                        <label class="form-label">Nombre del artículo</label>
                        <input type="text" name="product_name"
                               class="form-control"
                               value="<?= h($editProduct['name']) ?>" required>
                      </div>

                      <div class="mb-3">
                        <label class="form-label">Precio (MXN)</label>
                        <input type="number" step="0.01" min="0" name="product_price"
                               class="form-control"
                               value="<?= htmlspecialchars(number_format((float)$editProduct['price'], 2, '.', ''), ENT_QUOTES, 'UTF-8') ?>"
                               required>
                      </div>

                      <button class="btn btn-success w-100">Actualizar artículo</button>
                    </form>
                  <?php endif; ?>
                </div>

                <div class="col-12 col-lg-8">
                  <div class="card card-body border-0 shadow-sm h-100">
                    <h2 class="h6 mb-3">Artículos registrados</h2>
                    <?php if (!$productos): ?>
                      <p class="small text-muted mb-0">Aún no hay artículos registrados.</p>
                    <?php else: ?>
                      <div class="table-responsive" style="max-height: 380px; overflow:auto;">
                        <table class="table table-sm align-middle mb-0">
                          <thead>
                            <tr>
                              <th>ID</th>
                              <th>Artículo</th>
                              <th>Precio MXN</th>
                              <th>Estado</th>
                              <th>Alta</th>
                              <th></th>
                            </tr>
                          </thead>
                          <tbody>
                            <?php foreach ($productos as $prod): ?>
                              <tr>
                                <td><?= h($prod['id']) ?></td>
                                <td><?= h($prod['name']) ?></td>
                                <td>$<?= number_format((float)$prod['price'], 2) ?></td>
                                <td>
                                  <?php if ((int)$prod['is_active']): ?>
                                    <span class="badge bg-success badge-estado">Activo</span>
                                  <?php else: ?>
                                    <span class="badge bg-secondary badge-estado">Inactivo</span>
                                  <?php endif; ?>
                                </td>
                                <td class="small text-muted"><?= h($prod['created_at']) ?></td>
                                <td class="text-nowrap">
                                  <a href="userAdd.php?edit_product=<?= (int)$prod['id'] ?>#tab-articulos"
                                     class="btn btn-sm btn-outline-primary mb-1">
                                    Editar
                                  </a>
                                  <form method="post" class="d-inline">
                                    <input type="hidden" name="action" value="toggle_articulo">
                                    <input type="hidden" name="product_id" value="<?= (int)$prod['id'] ?>">
                                    <?php if ((int)$prod['is_active']): ?>
                                      <button class="btn btn-sm btn-outline-danger mb-1">Desactivar</button>
                                    <?php else: ?>
                                      <button class="btn btn-sm btn-outline-success mb-1">Activar</button>
                                    <?php endif; ?>
                                  </form>
                                </td>
                              </tr>
                            <?php endforeach; ?>
                          </tbody>
                        </table>
                      </div>
                    <?php endif; ?>
                  </div>
                </div>
              </div>
            </div>

            <!-- ================= TAB: RESET PASSWORD ================= -->
            <div class="tab-pane fade" id="tab-reset" role="tabpanel">
              <div class="card card-body border-0 shadow-sm">
                <h2 class="h6 mb-2">Reset de contraseña</h2>
                <p class="small text-muted">
                  Desde aquí puedes abrir el módulo independiente de recuperación / cambio de contraseñas.
                </p>
                <a href="reset_password.php" class="btn btn-outline-primary" target="_blank" rel="noopener">
                  Abrir módulo de reset de contraseña
                </a>
              </div>
            </div>

          </div><!-- /.tab-content -->
        </div><!-- /.card-body -->
      </div><!-- /.card -->

    </div><!-- /.col -->
  </div><!-- /.row -->
</div><!-- /.container-fluid -->

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
<script>
document.addEventListener('DOMContentLoaded', function () {
  if (window.location.hash === '#tab-articulos') {
    var triggerEl = document.querySelector('#tab-articulos-tab');
    if (triggerEl && typeof bootstrap !== 'undefined') {
      var tab = new bootstrap.Tab(triggerEl);
      tab.show();
    }
  }
});
</script>
</body>
</html>
