<?php
// api/ventas_update_pedido.php
// Actualiza OC y estatus de un pedido desde el rol "ventas"
// y regresa opcionalmente un link de WhatsApp para notificar al cliente.

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

header('Content-Type: application/json; charset=utf-8');

require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/auth.php';

$jsonOpts = JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES;

// ---- Validar sesión y rol ----
$u = current_user();
if (!$u || ($u['role'] ?? '') !== 'ventas') {
    http_response_code(403);
    echo json_encode(['ok' => false, 'error' => 'Acceso no autorizado'], $jsonOpts);
    exit;
}

// ---- Leer JSON de entrada ----
$input = json_decode(file_get_contents('php://input'), true);
if (!is_array($input)) {
    $input = [];
}

$order_id    = isset($input['order_id']) ? (int)$input['order_id'] : 0;
$status      = trim($input['status'] ?? '');
$sales_order = trim($input['sales_order'] ?? '');

$validStatuses = ['pendiente', 'autorizada', 'cancelada'];

if ($order_id <= 0 || !in_array($status, $validStatuses, true)) {
    http_response_code(400);
    echo json_encode(['ok' => false, 'error' => 'Datos inválidos'], $jsonOpts);
    exit;
}

try {
    $pdo = pdo();

    // Si queda AUTORIZADO y la OC viene vacía => generar SW-#ID
    if ($status === 'autorizada' && $sales_order === '') {
        $sales_order = 'SW-' . $order_id;
    }

    // === ACTUALIZAR PEDIDO ===
    // OJO: quitamos "updated_at = NOW()" por si esa columna no existe en tu tabla.
    // Si la tienes y se llama distinto, luego la volvemos a agregar ya con el nombre correcto.
    $stmt = $pdo->prepare("
        UPDATE orders
           SET status      = ?,
               sales_order = ?
         WHERE id = ?
    ");
    $stmt->execute([$status, $sales_order, $order_id]);

    // ====== Construir link de WhatsApp para el CLIENTE (opcional) ======
    $waLink = null;

    if ($status === 'autorizada') {
        // Tomamos datos del pedido + cliente
        $stmtInfo = $pdo->prepare("
            SELECT o.id,
                   o.delivery_date,
                   o.qty,
                   o.sales_order,
                   u.name  AS cliente,
                   u.phone AS phone
              FROM orders o
              JOIN users  u ON u.id = o.user_id
             WHERE o.id = ?
        ");
        $stmtInfo->execute([$order_id]);
        $row = $stmtInfo->fetch(PDO::FETCH_ASSOC);

        if ($row && !empty($row['phone'])) {
            $cliente   = $row['cliente'];
            $phone     = preg_replace('/\D+/', '', $row['phone']); // solo dígitos
            $delivery  = $row['delivery_date'];
            $qty       = $row['qty'];
            $oc        = $row['sales_order'];

            $mensaje = "Hola {$cliente}, tu pedido de hielo #{$order_id} ha sido AUTORIZADO.\n\n"
                     . "OC: {$oc}\n"
                     . "Entrega: {$delivery}\n"
                     . "Bloques: {$qty}\n\n"
                     . "Gracias por tu preferencia.";

            $waMsg  = rawurlencode($mensaje);
            $waLink = "https://wa.me/52{$phone}?text={$waMsg}";
        }
    }

    echo json_encode([
        'ok'          => true,
        'order_id'    => $order_id,
        'status'      => $status,
        'sales_order' => $sales_order,
        'wa_link'     => $waLink
    ], $jsonOpts);

} catch (Throwable $e) { // <-- capturamos cualquier error (Exception / Error)
    http_response_code(500);
    echo json_encode([
        'ok'     => false,
        'error'  => 'EXCEPTION',
        'detail' => $e->getMessage()
    ], $jsonOpts);
}
