<?php
// includes/auth.php
// Autenticación / sesión para Ventas Blanquita (MX/US + normalización E.164)

declare(strict_types=1);

require_once __DIR__ . '/db.php';

/** Asegura sesión activa */
function auth_ensure_session(): void {
    if (session_status() !== PHP_SESSION_ACTIVE) {
        session_start();
    }
}

/**
 * Helpers de esquema: validar si existe una columna (cache interno).
 */
function auth_table_has_column(PDO $pdo, string $table, string $column): bool {
    static $cache = [];
    $k = strtolower($table . '.' . $column);
    if (array_key_exists($k, $cache)) return (bool)$cache[$k];

    $stmt = $pdo->prepare("
        SELECT COUNT(*)
        FROM information_schema.columns
        WHERE table_schema = DATABASE()
          AND table_name = ?
          AND column_name = ?
    ");
    $stmt->execute([$table, $column]);
    $ok = ((int)$stmt->fetchColumn() > 0);
    $cache[$k] = $ok;
    return $ok;
}

/**
 * Normaliza un teléfono a formato E.164 para MX/US.
 * - MX: +521 + últimos 10 dígitos
 * - US: +1 + 10 dígitos
 * Si ya viene como +..., lo respeta pero corrige casos comunes (+52 -> +521).
 */
function auth_normalize_phone_e164(string $country, string $raw): string {
    $country = strtoupper(trim($country ?: 'MX'));
    $raw = trim((string)$raw);
    if ($raw === '') return '';

    // Si ya trae +, intentamos normalizar MX si trae +52 sin 1
    if (strpos($raw, '+') === 0) {
        $digits = preg_replace('/\D/', '', $raw) ?? '';
        if ($digits === '') return '';

        // +52xxxxxxxxxx => forzar +521 + last10
        if ($country === 'MX') {
            // Si viene +521 ya ok
            if (strpos($digits, '521') === 0 && strlen($digits) >= 13) {
                return '+'.$digits;
            }
            // Si viene +52 (o +520...) fuerza last10
            if (strpos($digits, '52') === 0 && strlen($digits) >= 12) {
                $last10 = substr($digits, -10);
                return '+521' . $last10;
            }
            // Si viene raro pero trae >=10 dígitos, toma last10
            if (strlen($digits) >= 10) {
                return '+521' . substr($digits, -10);
            }
        }

        // US: +1xxxxxxxxxx => ok si 11 total y empieza 1
        if ($country === 'US') {
            if (strpos($digits, '1') === 0 && strlen($digits) === 11) {
                return '+'.$digits;
            }
            // si trae 10 dígitos sin 1 pero venía con +, lo rearmamos
            if (strlen($digits) === 10) {
                return '+1'.$digits;
            }
        }

        // Si no podemos deducir, regresamos el +digits
        return '+'.$digits;
    }

    // Si NO trae +, limpiamos a solo dígitos
    $digits = preg_replace('/\D/', '', $raw) ?? '';
    $digits = trim($digits);
    if ($digits === '') return '';

    // Quita prefijos antiguos MX 044/045
    if ($country === 'MX' && (strpos($digits, '044') === 0 || strpos($digits, '045') === 0)) {
        $digits = substr($digits, 3);
    }

    if ($country === 'MX') {
        if (strlen($digits) < 10) return '';
        $last10 = substr($digits, -10);
        return '+521' . $last10;
    }

    if ($country === 'US') {
        // si viene 11 con leading 1
        if (strlen($digits) === 11 && $digits[0] === '1') {
            $digits = substr($digits, 1);
        }
        if (strlen($digits) !== 10) return '';
        return '+1' . $digits;
    }

    return '';
}

/**
 * Devuelve usuario actual desde sesión, o null.
 * Estructura: [id, name, phone, role, address]
 */
function current_user(): ?array {
    auth_ensure_session();
    return $_SESSION['user'] ?? null;
}

/** Cierra sesión */
function logout_user(): void {
    auth_ensure_session();
    $_SESSION = [];
    if (ini_get('session.use_cookies')) {
        $p = session_get_cookie_params();
        setcookie(session_name(), '', time() - 42000, $p['path'], $p['domain'], (bool)$p['secure'], (bool)$p['httponly']);
    }
    session_destroy();
}

/**
 * Login por teléfono + contraseña
 * - Si $phone no viene en +E164 y envías $country (MX/US), lo normaliza.
 */
function login(string $phone, string $password, string $country = 'MX'): bool {
    auth_ensure_session();

    $phone = trim($phone);
    if ($phone === '' || $password === '') return false;

    // Si no viene en +E164, lo normalizamos (default MX)
    if (strpos($phone, '+') !== 0) {
        $phone = auth_normalize_phone_e164($country, $phone);
    }

    if ($phone === '' || strpos($phone, '+') !== 0) return false;

    $pdo  = pdo();
    $stmt = $pdo->prepare('SELECT id, name, phone, role, address, password_hash FROM users WHERE phone = ? LIMIT 1');
    $stmt->execute([$phone]);
    $u = $stmt->fetch(PDO::FETCH_ASSOC);

    if ($u && isset($u['password_hash']) && password_verify($password, (string)$u['password_hash'])) {
        $_SESSION['user'] = [
            'id'      => (int)$u['id'],
            'name'    => (string)$u['name'],
            'phone'   => (string)$u['phone'],
            'role'    => (string)$u['role'],
            'address' => (string)($u['address'] ?? ''),
        ];
        return true;
    }
    return false;
}

/**
 * Crea usuario.
 * - Normaliza phone a +E164 según country (MX/US)
 * - Si existen columnas phone_country y phone_national, las llena (sin romper si aún no existen)
 *
 * Parámetros nuevos opcionales:
 *   $country: 'MX' | 'US'
 *   $phone_raw: el valor original del formulario (solo para guardar nacional)
 */
function register_user(
    string $name,
    string $phone,
    string $address,
    string $password,
    string $cp = '',
    string $role = 'cliente',
    string $country = 'MX',
    string $phone_raw = ''
): int {
    $name  = trim($name);
    $phone = trim($phone);
    $country = strtoupper(trim($country ?: 'MX'));

    if ($name === '' || $phone === '' || $password === '') {
        throw new InvalidArgumentException('Datos incompletos');
    }

    // Normaliza SIEMPRE a +E164 antes de guardar
    $phoneE164 = (strpos($phone, '+') === 0) ? auth_normalize_phone_e164($country, $phone) : auth_normalize_phone_e164($country, $phone);
    if ($phoneE164 === '' || strpos($phoneE164, '+') !== 0) {
        throw new InvalidArgumentException('Teléfono inválido');
    }

    $pdo  = pdo();
    $hash = password_hash($password, PASSWORD_DEFAULT);

    // Inserción base (tu esquema actual usa cp)
    $stmt = $pdo->prepare(
        "INSERT INTO users (name, phone, address, cp, password_hash, role, created_at)
         VALUES (?, ?, ?, ?, ?, ?, NOW())"
    );
    $stmt->execute([$name, $phoneE164, $address, $cp, $hash, $role]);

    $id = (int)$pdo->lastInsertId();

    // Guardado opcional de país/nacional si columnas existen
    try {
        $hasCountry  = auth_table_has_column($pdo, 'users', 'phone_country');
        $hasNational = auth_table_has_column($pdo, 'users', 'phone_national');

        if ($hasCountry || $hasNational) {
            $digitsRaw = preg_replace('/\D/', '', (string)$phone_raw) ?? '';
            $digitsRaw = trim($digitsRaw);
            $national = '';
            if ($digitsRaw !== '') {
                // Guardamos "nacional" como últimos 10 si hay, si no, lo que haya
                $national = (strlen($digitsRaw) >= 10) ? substr($digitsRaw, -10) : $digitsRaw;
            } else {
                // Si no hay raw, intentamos derivarlo del E164 (últimos 10)
                $digitsE = preg_replace('/\D/', '', (string)$phoneE164) ?? '';
                $digitsE = trim($digitsE);
                $national = (strlen($digitsE) >= 10) ? substr($digitsE, -10) : $digitsE;
            }

            $sets = [];
            $vals = [];

            if ($hasCountry)  { $sets[] = "phone_country = ?";  $vals[] = $country; }
            if ($hasNational) { $sets[] = "phone_national = ?"; $vals[] = $national; }

            if ($sets) {
                $vals[] = $id;
                $sql = "UPDATE users SET ".implode(", ", $sets)." WHERE id = ?";
                $pdo->prepare($sql)->execute($vals);
            }
        }
    } catch (Throwable $e) {
        // No rompe el registro si aún no agregas columnas
    }

    return $id;
}

/**
 * Requiere sesión (y opcionalmente roles). Para vistas (no API).
 */
function require_login(array $roles = []): array {
    $u = current_user();
    if (!$u) {
        header('Location: index.php');
        exit;
    }
    if ($roles && !in_array(($u['role'] ?? ''), $roles, true)) {
        http_response_code(403);
        echo 'No autorizado';
        exit;
    }
    return $u;
}
