<?php
// includes/sms_outbox.php
// Cola (outbox) para SMS: reintentos y envío desacoplado.

declare(strict_types=1);

require_once __DIR__ . '/db.php';
require_once __DIR__ . '/../sms_helper.php';

/**
 * Canonicaliza E.164 para que el UNIQUE no falle por formatos distintos.
 * - deja solo '+' y dígitos
 * - garantiza que empiece con '+'
 */
function sms_canon_e164(string $toE164): string {
    $toE164 = trim($toE164);
    if ($toE164 === '') return '';
    $s = preg_replace('/[^0-9\+]/', '', $toE164) ?? '';
    $s = trim($s);
    if ($s === '') return '';
    // Si no inicia con + pero parece internacional, intenta prefijar
    if ($s[0] !== '+') $s = '+' . preg_replace('/\D/', '', $s);
    // Asegurar '+<digits>'
    $d = preg_replace('/\D/', '', $s) ?? '';
    if (strlen($d) < 8) return '';
    return '+' . $d;
}

/** Asegura que exista la tabla (solo 1 vez por request) */
function sms_ensure_outbox_table(PDO $pdo): void {
    static $done = false;
    if ($done) return;
    $done = true;

    // InnoDB recomendado para consistencia / concurrencia.
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS `sms_outbox` (
          `id` int(11) NOT NULL AUTO_INCREMENT,
          `event_type` varchar(50) NOT NULL,
          `order_id` int(11) NOT NULL,
          `to_phone` varchar(30) NOT NULL,
          `body` text NOT NULL,
          `status` enum('pending','sent','failed') NOT NULL DEFAULT 'pending',
          `attempts` int(11) NOT NULL DEFAULT 0,
          `last_error` varchar(255) DEFAULT NULL,
          `created_at` datetime NOT NULL DEFAULT current_timestamp(),
          `last_try_at` datetime DEFAULT NULL,
          `sent_at` datetime DEFAULT NULL,
          PRIMARY KEY (`id`),
          UNIQUE KEY `uniq_event_order_to` (`event_type`,`order_id`,`to_phone`),
          KEY `idx_status_attempts` (`status`,`attempts`),
          KEY `idx_created` (`created_at`),
          KEY `idx_order` (`order_id`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    ");
}

/** Obtiene teléfonos (E.164) de usuarios por rol */
function sms_get_role_tos(PDO $pdo, string $role): array {

    // Intentar con phone_country e is_active; si no existen, degradar.
    $rows = [];
    $okPhoneCountry = true;
    $okIsActive = true;

    try {
        // Prueba con todo
        $stmt = $pdo->prepare("SELECT phone, phone_country, is_active FROM users WHERE role = ?");
        $stmt->execute([$role]);
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
    } catch (Throwable $e) {
        // Si falla, probamos sin phone_country o sin is_active en cascada
        $rows = [];
        $okPhoneCountry = false;
        $okIsActive = false;
    }

    // Si la primera consulta falló, intentamos combinaciones más simples:
    if (!$rows) {
        try {
            $stmt = $pdo->prepare("SELECT phone, phone_country FROM users WHERE role = ? AND is_active = 1");
            $stmt->execute([$role]);
            $rows = $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
            $okPhoneCountry = true;
            $okIsActive = true;
        } catch (Throwable $e) {
            $okPhoneCountry = false;
        }
    }

    if (!$rows) {
        try {
            $stmt = $pdo->prepare("SELECT phone FROM users WHERE role = ? AND is_active = 1");
            $stmt->execute([$role]);
            $tmp = $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
            $rows = array_map(function($r){
                return ['phone' => $r['phone'] ?? '', 'phone_country' => 'MX', 'is_active' => 1];
            }, $tmp);
            $okIsActive = true;
        } catch (Throwable $e) {
            $okIsActive = false;
        }
    }

    if (!$rows) {
        // Último fallback: solo role
        $stmt = $pdo->prepare("SELECT phone FROM users WHERE role = ?");
        $stmt->execute([$role]);
        $tmp = $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
        $rows = array_map(function($r){
            return ['phone' => $r['phone'] ?? '', 'phone_country' => 'MX', 'is_active' => 1];
        }, $tmp);
    }

    $tos = [];
    foreach ($rows as $r) {
        // Respeta is_active si existe
        if ($okIsActive) {
            $active = (int)($r['is_active'] ?? 1);
            if ($active !== 1) continue;
        }

        $raw = (string)($r['phone'] ?? '');
        $cc  = $okPhoneCountry ? (string)($r['phone_country'] ?? 'MX') : 'MX';

        // Si ya viene +521..., sms_phone_to_e164 lo respeta.
        $e164 = sms_phone_to_e164($raw, $cc ?: 'MX');
        $e164 = sms_canon_e164($e164);
        if ($e164 !== '') $tos[] = $e164;
    }

    return array_values(array_unique($tos));
}

/** Inserta en outbox si no existe (idempotencia por event_type/order/to) */
function sms_queue(PDO $pdo, string $eventType, int $orderId, string $toE164, string $body): bool {
    $eventType = trim($eventType);
    $toE164    = sms_canon_e164($toE164);
    $body      = trim($body);

    if ($eventType === '' || $orderId <= 0 || $toE164 === '' || $body === '') return false;

    // Si sms está deshabilitado, no hacemos nada.
    if (!sms_available()) return false;

    sms_ensure_outbox_table($pdo);

    $stmt = $pdo->prepare("INSERT IGNORE INTO sms_outbox(event_type, order_id, to_phone, body) VALUES(?,?,?,?)");
    return (bool)$stmt->execute([$eventType, $orderId, $toE164, $body]);
}

function sms_queue_to_role(PDO $pdo, string $role, string $eventType, int $orderId, string $body): array {
    $tos = sms_get_role_tos($pdo, $role);
    $queued = 0;
    foreach ($tos as $to) {
        if (sms_queue($pdo, $eventType, $orderId, $to, $body)) $queued++;
    }
    return ['role' => $role, 'queued' => $queued, 'recipients' => count($tos)];
}

function sms_queue_to_phone(PDO $pdo, string $rawPhone, string $eventType, int $orderId, string $body, string $country = 'MX'): array {
    $to = sms_phone_to_e164($rawPhone, $country);
    $to = sms_canon_e164($to);

    if ($to === '') return ['queued' => 0, 'recipients' => 0, 'error' => 'INVALID_PHONE'];

    $ok = sms_queue($pdo, $eventType, $orderId, $to, $body);
    return ['queued' => $ok ? 1 : 0, 'recipients' => 1, 'to' => $to];
}

/**
 * Procesa outbox:
 * - Selecciona "pending" con attempts < maxAttempts
 * - Incrementa attempts, registra last_try_at
 * - Envía y marca sent/failed
 *
 * Nota: si ejecutas 2 workers al mismo tiempo, puede haber doble envío.
 * En la práctica, evita correr más de 1 cron simultáneo.
 */
function sms_process_outbox(PDO $pdo, int $limit = 20, int $maxAttempts = 5): array {
    if (!sms_available()) return ['processed' => 0, 'sent' => 0, 'failed' => 0, 'disabled' => true];

    sms_ensure_outbox_table($pdo);

    $limit = max(1, min(200, (int)$limit));
    $maxAttempts = max(1, min(20, (int)$maxAttempts));

    $rows = $pdo->query("
        SELECT id, to_phone, body, attempts
        FROM sms_outbox
        WHERE status = 'pending' AND attempts < {$maxAttempts}
        ORDER BY created_at ASC
        LIMIT {$limit}
    ")->fetchAll(PDO::FETCH_ASSOC);

    $processed = 0; $sent = 0; $failed = 0;

    foreach ($rows as $r) {
        $processed++;
        $id   = (int)$r['id'];
        $to   = sms_canon_e164((string)$r['to_phone']);
        $body = (string)$r['body'];

        // Marcar intento
        $pdo->prepare("UPDATE sms_outbox SET attempts = attempts + 1, last_try_at = NOW() WHERE id = ?")
            ->execute([$id]);

        try {
            $res = sms_send_one($to, $body);

            if (!($res['ok'] ?? false)) {
                $err = substr((string)($res['error'] ?? 'SEND_FAILED'), 0, 200);
                $det = isset($res['detail']) ? (' - ' . (string)$res['detail']) : '';
                $sdk = isset($res['sdk_error']) && $res['sdk_error'] ? (' | SDK: ' . (string)$res['sdk_error']) : '';
                $msg = substr($err . $det . $sdk, 0, 250);

                $pdo->prepare("UPDATE sms_outbox SET last_error = ? WHERE id = ?")->execute([$msg, $id]);

                // Si ya llegó al máximo, marcar failed
                $pdo->prepare("UPDATE sms_outbox SET status = IF(attempts >= ?, 'failed', status) WHERE id = ?")
                    ->execute([$maxAttempts, $id]);

                $failed++;
                continue;
            }

            $pdo->prepare("UPDATE sms_outbox SET status = 'sent', sent_at = NOW(), last_error = NULL WHERE id = ?")
                ->execute([$id]);

            $sent++;

        } catch (Throwable $e) {
            $pdo->prepare("UPDATE sms_outbox SET last_error = ? WHERE id = ?")
                ->execute([substr($e->getMessage(), 0, 250), $id]);

            $pdo->prepare("UPDATE sms_outbox SET status = IF(attempts >= ?, 'failed', status) WHERE id = ?")
                ->execute([$maxAttempts, $id]);

            $failed++;
        }
    }

    return ['processed' => $processed, 'sent' => $sent, 'failed' => $failed];
}
