<?php
// includes/wa_outbox.php
// Cola/worker de notificaciones WhatsApp (Twilio)
// Compatible con esquema: attempts, last_try_at, last_error
// PHP 8.1+

declare(strict_types=1);

require_once __DIR__ . '/db.php';

$waHelper = dirname(__DIR__) . '/whatsapp_helper.php';
if (is_file($waHelper)) require_once $waHelper;

function wa_available_configured(): bool {
    if (!function_exists('wa_config') || !function_exists('wa_available')) return false;
    if (!wa_available()) return false;

    $cfg = wa_config();
    $sid   = trim((string)($cfg['sid'] ?? ''));
    $token = trim((string)($cfg['token'] ?? ''));
    $from  = trim((string)($cfg['from'] ?? ''));
    $msSid = trim((string)($cfg['messaging_service_sid'] ?? ''));

    // Con Messaging Service SID, 'from' puede ir vacío
    $hasFromOrMs = ($from !== '') || ($msSid !== '');
    return ($sid !== '' && $token !== '' && $hasFromOrMs);
}

function wa_db_has_column(PDO $pdo, string $table, string $column): bool {
    static $cache = [];
    $key = strtolower($table . '.' . $column);
    if (array_key_exists($key, $cache)) return (bool)$cache[$key];

    $st = $pdo->prepare("
        SELECT COUNT(*)
        FROM information_schema.columns
        WHERE table_schema = DATABASE()
          AND table_name = ?
          AND column_name = ?
    ");
    $st->execute([$table, $column]);
    $cache[$key] = ((int)$st->fetchColumn() > 0);
    return (bool)$cache[$key];
}

/**
 * Encola un mensaje (INSERT IGNORE) usando tu UNIQUE(event_type, order_id, to_whatsapp).
 * Inserción mínima para evitar diferencias de nombres de columnas.
 */
function wa_queue(PDO $pdo, string $toWhatsApp, string $body, string $eventType, int $orderId): bool {
    $toWhatsApp = trim($toWhatsApp);
    $body       = trim($body);
    $eventType  = trim($eventType);

    if ($toWhatsApp === '' || $body === '' || $eventType === '' || $orderId <= 0) return false;

    if (function_exists('wa_phone_to_whatsapp')) {
        $toWhatsApp = wa_phone_to_whatsapp($toWhatsApp, 'MX');
    } else {
        if (stripos($toWhatsApp, 'whatsapp:') !== 0) $toWhatsApp = 'whatsapp:' . $toWhatsApp;
    }

    if ($toWhatsApp === '' || stripos($toWhatsApp, 'whatsapp:') !== 0) return false;

    $stmt = $pdo->prepare("
        INSERT IGNORE INTO wa_outbox (to_whatsapp, body, event_type, order_id)
        VALUES (?, ?, ?, ?)
    ");
    return (bool)$stmt->execute([$toWhatsApp, $body, $eventType, $orderId]);
}

function wa_queue_to_phone(PDO $pdo, string $phone, string $eventType, int $orderId, string $body, string $countryHint = 'MX'): bool {
    $phone = trim($phone);
    if ($phone === '') return false;

    $to = function_exists('wa_phone_to_whatsapp')
        ? wa_phone_to_whatsapp($phone, $countryHint)
        : ('whatsapp:' . $phone);

    if ($to === '' || stripos($to, 'whatsapp:') !== 0) return false;
    return wa_queue($pdo, $to, $body, $eventType, $orderId);
}

function wa_get_role_tos(PDO $pdo, string $role): array {
    $role = trim($role);
    if ($role === '') return [];

    $hasActive  = wa_db_has_column($pdo, 'users', 'is_active');
    $hasCountry = wa_db_has_column($pdo, 'users', 'phone_country');

    $sql = "SELECT DISTINCT phone" . ($hasCountry ? ", phone_country" : "") . "
            FROM users
            WHERE role = ?
              AND phone IS NOT NULL AND phone <> ''";
    if ($hasActive) $sql .= " AND is_active = 1";

    $st = $pdo->prepare($sql);
    $st->execute([$role]);
    $rows = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];

    $out = [];
    foreach ($rows as $r) {
        $raw = trim((string)($r['phone'] ?? ''));
        if ($raw === '') continue;

        $country = $hasCountry ? strtoupper(trim((string)($r['phone_country'] ?? 'MX'))) : 'MX';

        $to = function_exists('wa_phone_to_whatsapp')
            ? wa_phone_to_whatsapp($raw, $country)
            : ('whatsapp:' . $raw);

        if ($to !== '' && stripos($to, 'whatsapp:') === 0) $out[] = $to;
    }
    return array_values(array_unique($out));
}

function wa_queue_to_role(PDO $pdo, string $role, string $eventType, int $orderId, string $body): array {
    $tos  = wa_get_role_tos($pdo, $role);
    $ok   = 0;
    $fail = 0;

    foreach ($tos as $to) {
        if (wa_queue($pdo, $to, $body, $eventType, $orderId)) $ok++;
        else $fail++;
    }

    return ['ok' => $ok, 'fail' => $fail, 'count' => count($tos)];
}

/**
 * Procesa outbox:
 * - Lee pending
 * - Envía Twilio
 * - Si OK -> sent
 * - Si falla -> failed + last_error
 *
 * Guarda en last_error:
 *   WA_SEND_FAILED | twilio_code=XXXX | status=YYY | detail=...
 */
function wa_process_outbox(PDO $pdo, int $limit = 10): array {
    $limit = max(1, min(200, $limit));

    if (!function_exists('wa_send_one_result')) {
        return ['sent' => 0, 'failed' => 0, 'skipped' => 0, 'reason' => 'wa_send_one_result_missing'];
    }
    if (!wa_available_configured()) {
        return ['sent' => 0, 'failed' => 0, 'skipped' => 0, 'reason' => 'wa_not_configured'];
    }

    $st = $pdo->prepare("
        SELECT id, to_whatsapp, body
        FROM wa_outbox
        WHERE status = 'pending'
        ORDER BY id ASC
        LIMIT {$limit}
    ");
    $st->execute();
    $rows = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];

    $sent = 0; $failed = 0; $skipped = 0;

    foreach ($rows as $r) {
        $id  = (int)($r['id'] ?? 0);
        $to  = (string)($r['to_whatsapp'] ?? '');
        $msg = (string)($r['body'] ?? '');

        if ($id <= 0 || $to === '' || $msg === '') {
            $skipped++;
            continue;
        }

        $res = wa_send_one_result($to, $msg, 'MX');

        if (!empty($res['ok'])) {
            $up = $pdo->prepare("
                UPDATE wa_outbox
                SET status='sent',
                    sent_at=NOW(),
                    last_try_at=NOW(),
                    attempts = attempts + 1,
                    last_error=NULL
                WHERE id=?
            ");
            $up->execute([$id]);
            $sent++;
            continue;
        }

        $twCode = (string)($res['twilio_code'] ?? '0');
        $twSt   = (string)($res['status'] ?? '0');
        $detail = trim((string)($res['detail'] ?? ($res['error'] ?? 'SEND_FAILED')));

        // Recortar detalle para evitar textos enormes
        if (strlen($detail) > 450) $detail = substr($detail, 0, 450) . '...';

        $err = "WA_SEND_FAILED | twilio_code={$twCode} | status={$twSt} | {$detail}";

        $up = $pdo->prepare("
            UPDATE wa_outbox
            SET status='failed',
                last_try_at=NOW(),
                attempts = attempts + 1,
                last_error=?
            WHERE id=?
        ");
        $up->execute([$err, $id]);
        $failed++;
    }

    return ['sent' => $sent, 'failed' => $failed, 'skipped' => $skipped, 'checked' => count($rows)];
}
