<?php
// sms_helper.php
// Envío de SMS con Twilio (preferente) con fallback HTTP (sin SDK).
// Portable: PHP 7.4+ (sin funciones exclusivas de PHP 8).

declare(strict_types=1);

/**
 * Configuración esperada (en includes/config.local.php o variables de entorno):
 *
 * ENV:
 * - TWILIO_ACCOUNT_SID
 * - TWILIO_AUTH_TOKEN
 * - TWILIO_SMS_FROM (Ej: +1415..., +52..., o Messaging Service SID MGxxxx)
 * - VB_SMS_ENABLED  (1/0) opcional
 *
 * En app_config():
 *  'twilio' => [
 *    'sid'             => 'AC...',
 *    'token'           => '...',
 *    'sms_from'        => '+1...'  // o 'MG...'
 *    'default_country' => '+52'    // opcional
 *  ],
 *  'sms' => [
 *    'enabled' => true,
 *  ],
 *  // opcional:
 *  // 'vendor_autoload' => '/home/USUARIO/public_html/vendor/autoload.php',
 */

function sms_starts_with(string $haystack, string $needle): bool {
    if ($needle === '') return true;
    return strncmp($haystack, $needle, strlen($needle)) === 0;
}

function sms_config(): array {
    $cfgAll = function_exists('app_config') ? (app_config() ?: []) : [];
    $tw     = is_array($cfgAll['twilio'] ?? null) ? $cfgAll['twilio'] : [];
    $sms    = is_array($cfgAll['sms'] ?? null) ? $cfgAll['sms'] : [];

    $sidEnv   = getenv('TWILIO_ACCOUNT_SID') ?: '';
    $tokenEnv = getenv('TWILIO_AUTH_TOKEN') ?: '';
    $fromEnv  = getenv('TWILIO_SMS_FROM') ?: '';

    $sid   = trim((string)($tw['sid']   ?? $sidEnv));
    $token = trim((string)($tw['token'] ?? $tokenEnv));
    $from  = trim((string)($tw['sms_from'] ?? $fromEnv));

    $enabledEnv = trim((string)(getenv('VB_SMS_ENABLED') ?: ''));
    $enabled = (bool)($sms['enabled'] ?? ($enabledEnv !== '' ? ($enabledEnv === '1') : true));

    // Para normalización (si llegas a usarla)
    $defaultCountry = trim((string)($tw['default_country'] ?? '+52'));
    if ($defaultCountry === '' || $defaultCountry[0] !== '+') $defaultCountry = '+52';

    // Autoload opcional global
    $vendorAutoload = trim((string)($cfgAll['vendor_autoload'] ?? ''));

    return [
        'enabled'         => $enabled,
        'sid'             => $sid,
        'token'           => $token,
        'from'            => $from,
        'default_country' => $defaultCountry,
        'vendor_autoload' => $vendorAutoload,
    ];
}

function sms_available(): bool {
    $c = sms_config();
    return $c['enabled'] && $c['sid'] !== '' && $c['token'] !== '' && $c['from'] !== '';
}

/**
 * Normaliza teléfono a E.164.
 * - Si ya viene con +, se respeta (limpia a +<digits>).
 * - MX:
 *    - 10 dígitos -> +52 + 10
 *    - 12 dígitos iniciando 52 -> +52 + 10
 *    - 13 dígitos iniciando 521 -> +521 + 10 (móvil “1”)
 * - Si no se puede inferir, devuelve vacío.
 */
function sms_phone_to_e164(string $raw, string $country = 'MX'): string {
    $raw = trim($raw);
    if ($raw === '') return '';

    $country = strtoupper(trim($country ?: 'MX'));

    // Extraer solo dígitos y +
    $digitsPlus = preg_replace('/[^0-9\+]/', '', $raw) ?? '';
    $digitsPlus = trim($digitsPlus);

    // Caso: ya viene con +
    if ($digitsPlus !== '' && $digitsPlus[0] === '+') {
        $d = preg_replace('/\D/', '', $digitsPlus) ?? '';
        if (strlen($d) < 8) return '';
        return '+' . $d;
    }

    // Solo dígitos
    $digits = preg_replace('/\D/', '', $raw) ?? '';
    if ($digits === '') return '';

    // México
    if ($country === 'MX') {
        if (strlen($digits) === 10) return '+52' . $digits;
        if (strlen($digits) === 12 && sms_starts_with($digits, '52')) return '+' . $digits;
        if (strlen($digits) === 13 && sms_starts_with($digits, '521')) return '+' . $digits;
    }

    // Para otros países, no adivinamos sin + (extiende si lo necesitas).
    return '';
}

/**
 * Encuentra un autoload usable para Twilio SDK:
 * - Config vendor_autoload (si existe)
 * - sms_helper.php/vendor/autoload.php (si sms_helper.php está en raíz del proyecto)
 * - ../vendor/autoload.php (si sms_helper.php está en subcarpeta)
 */
function sms_find_autoload_path(): string {
    $cfg = sms_config();

    $candidates = [];
    if (!empty($cfg['vendor_autoload'])) $candidates[] = $cfg['vendor_autoload'];

    $candidates[] = __DIR__ . '/vendor/autoload.php';
    $candidates[] = dirname(__DIR__) . '/vendor/autoload.php';

    foreach ($candidates as $p) {
        $p = (string)$p;
        if ($p !== '' && is_file($p)) return $p;
    }
    return '';
}

/**
 * Envía un SMS. Devuelve array con ok y metadatos.
 * - Si existe el SDK (autoload + Twilio\Rest\Client), lo usa.
 * - Si no, usa HTTP POST a la API de Twilio (requiere cURL o allow_url_fopen).
 */
function sms_send_one(string $toE164, string $body): array {
    $cfg = sms_config();

    if (!$cfg['enabled']) return ['ok' => false, 'error' => 'SMS_DISABLED'];
    if ($cfg['sid'] === '' || $cfg['token'] === '' || $cfg['from'] === '') {
        return ['ok' => false, 'error' => 'SMS_CONFIG_INCOMPLETE'];
    }

    $toE164 = trim($toE164);
    if ($toE164 === '') return ['ok' => false, 'error' => 'INVALID_TO'];

    $body = trim($body);
    if ($body === '') return ['ok' => false, 'error' => 'EMPTY_BODY'];

    // Control de longitud (Twilio segmenta, pero evitamos abusos)
    if (strlen($body) > 1200) $body = substr($body, 0, 1200) . '…';

    $lastSdkError = null;

    // Preferir SDK si existe
    $autoload = sms_find_autoload_path();
    if ($autoload !== '') {
        try {
            require_once $autoload;

            if (class_exists('Twilio\\Rest\\Client')) {
                $client = new Twilio\Rest\Client($cfg['sid'], $cfg['token']);

                // Nota: EN SDK, el "to" va como primer argumento.
                $params = ['body' => $body];

                // from puede ser número E.164 o MessagingServiceSid (MGxxxx)
                if (sms_starts_with($cfg['from'], 'MG')) {
                    $params['messagingServiceSid'] = $cfg['from'];
                } else {
                    $params['from'] = $cfg['from'];
                }

                $msg = $client->messages->create($toE164, $params);
                return ['ok' => true, 'sid' => (string)$msg->sid, 'via' => 'sdk'];
            }
        } catch (Throwable $e) {
            $lastSdkError = $e->getMessage();
            // caemos a HTTP fallback
        }
    }

    // HTTP fallback
    $url = 'https://api.twilio.com/2010-04-01/Accounts/' . rawurlencode($cfg['sid']) . '/Messages.json';
    $post = [
        'To'   => $toE164,
        'Body' => $body,
    ];
    if (sms_starts_with($cfg['from'], 'MG')) {
        $post['MessagingServiceSid'] = $cfg['from'];
    } else {
        $post['From'] = $cfg['from'];
    }

    $auth = base64_encode($cfg['sid'] . ':' . $cfg['token']);

    // cURL
    if (function_exists('curl_init')) {
        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($post));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Authorization: Basic ' . $auth,
            'Content-Type: application/x-www-form-urlencoded',
        ]);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 20);

        $resp = curl_exec($ch);
        $err  = curl_error($ch);
        $code = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($resp === false) {
            return [
                'ok' => false,
                'error' => 'CURL_ERROR',
                'detail' => $err,
                'sdk_error' => $lastSdkError,
            ];
        }

        $json = json_decode((string)$resp, true);
        if ($code >= 200 && $code < 300) {
            return ['ok' => true, 'sid' => (string)($json['sid'] ?? ''), 'via' => 'http'];
        }

        return [
            'ok' => false,
            'error' => 'TWILIO_HTTP',
            'http' => $code,
            'detail' => (string)($json['message'] ?? $resp),
            'sdk_error' => $lastSdkError,
        ];
    }

    // allow_url_fopen fallback
    $ctx = stream_context_create([
        'http' => [
            'method'  => 'POST',
            'header'  => "Authorization: Basic {$auth}\r\nContent-Type: application/x-www-form-urlencoded\r\n",
            'content' => http_build_query($post),
            'timeout' => 20,
        ],
    ]);

    $resp = @file_get_contents($url, false, $ctx);

    // Obtener status code si está disponible
    $status = 0;
    if (isset($http_response_header) && is_array($http_response_header) && isset($http_response_header[0])) {
        if (preg_match('/\s(\d{3})\s/', (string)$http_response_header[0], $m)) {
            $status = (int)$m[1];
        }
    }

    if ($resp === false) {
        return [
            'ok' => false,
            'error' => 'HTTP_FOPEN_FAILED',
            'http' => $status,
            'sdk_error' => $lastSdkError,
        ];
    }

    $json = json_decode((string)$resp, true);

    if ($status >= 200 && $status < 300) {
        return ['ok' => true, 'sid' => (string)($json['sid'] ?? ''), 'via' => 'http_fopen'];
    }

    return [
        'ok' => false,
        'error' => 'TWILIO_HTTP',
        'http' => $status,
        'detail' => (string)($json['message'] ?? $resp),
        'sdk_error' => $lastSdkError,
    ];
}
